#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# dbdatasources.py  -  database datasources for datasourceProvider.py
#
# $Revision: 2.10 $
#
# Copyright (C) 2016 PerFact Innovation GmbH & Co. KG <info@perfact.de>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#


import time
import ast
import os
import six

if not six.PY2:
    basestring = str

# os.environ['PGCLIENTENCODING'] = 'LATIN1'
os.environ['PGCLIENTENCODING'] = 'UNICODE'


literal_controlvalues = {'controlfloat': '1.0001', 'controlstring': "'foo'"}
'''Literal control values are querried as literals from the remote
system. Then they get compared with the same literal values in the
local database. This is a way to make sure data is transfered
correctly.'''


class ConnectionDecorator(object):
    ''' Decorator wrapping database connections'''

    def __init__(self, connector, dbtype='', cacheconn_id=None):
        self.connector = connector
        self.decorated = None
        self.dbtype = dbtype
        self.cacheconn_id = cacheconn_id
        self.__phantom_table = ''
        self.initProvider()
        if self.dbtype[-2:] == '_j':
            # work with julian date
            self.nowstring = self.julian_nowstring
        if self.dbtype[-4:] == '_std':
            # work with standard sql
            self.__phantom_table = "(values ('foo')) t"
        if 'oracle' in self.dbtype:
            # oracle flavoured database connection
            self.__phantom_table = 'DUAL'

    def initProvider(self):
        ''' Do all the provider specific stuff'''
        if self.dbtype == 'oracle':
            import cx_Oracle
            self.provider = cx_Oracle
            # os.environ["NLS_LANG"] = 'German_Germany.UTF8'
        elif self.dbtype == 'sapdb':
            self.connector = ast.literal_eval(self.connector)
            import sapdb.dbapi
            self.provider = sapdb.dbapi
            self.close = self.noClose
        elif self.dbtype.startswith('pyodbc'):
            # this way we can flavour pyodbc using '_j', '_std' or '_oracle'
            import pyodbc
            self.provider = pyodbc
        elif self.dbtype == 'psycopg':
            import psycopg2
            self.provider = psycopg2
        elif self.dbtype.startswith('sybase'):
            raise Exception('No more support for Sybase. Try pyodbc.')
        elif self.dbtype.startswith('dbdummy'):
            self.provider = None
        elif self.dbtype == 'mqtt':
            # mqtt is just another dummy database type
            self.provider = None
        else:
            raise NotImplementedError('Invalid database type provided')

    def __del__(self):
        self.close()

    def create_connection(self):
        try:
            if isinstance(self.connector, basestring):
                self.decorated = self.provider.connect(self.connector)
            else:
                self.decorated = self.provider.connect(**self.connector)
        except Exception as error:
            raise dbconnectorException(
                self.dbtype,
                'cacheconn_id={}'.format(self.cacheconn_id),
                error
            )

    def cursor(self):
        if self.decorated is None:
            self.create_connection()
        return self.decorated.cursor()

    def close(self):
        if self.decorated is not None:
            try:
                self.decorated.close()
            except Exception:
                pass
            self.decorated = None

    def noClose(self):
        ''' Substitute for close '''
        return

    def nowstring(self):
        return 'now()'

    def julian_nowstring(self):
        ''' Substitute for nowstring'''
        return str(int(time.strftime('%Y%j'))-1900000)

    def set_isolation_level(self, level):
        if (self.decorated is None):
            self.create_connection()
        return self.decorated.set_isolation_level(level)

    def commit(self):
        return self.decorated.commit()

    def rollback(self):
        if self.decorated is None:
            return
        else:
            return self.decorated.rollback()

    def duplicate(self):
        return ConnectionDecorator(
            dbtype=self.dbtype, connector=self.connector)

    @property
    def phantom_table(self):
        return self.__phantom_table

    def create_control_statement(self):
        ''' create an sql statement for the controlvalues '''
        # create a statement with the controlvalues
        pairs = []
        for n, v in six.iteritems(literal_controlvalues):
            pairs.append(v+' as '+n)
        from_part = ''
        # some dbs have a special phantom table. ConnectionDecorator knows.
        if self.__phantom_table:
            from_part = ' from ' + self.__phantom_table
        statement = 'select ' + ','.join(pairs) + from_part
        return statement

    def query_controlvalues(self):
        ''' get some literal values from the db '''
        statement = self.create_control_statement()
        # query and get description and remote_data
        cur = self.cursor()
        cur.execute(statement)
        desc = cur.description
        remote_data = cur.fetchall()
        cur.close()
        assert len(desc) == len(literal_controlvalues), \
            'Not all controlvalues returned'

        colnames = []
        for col in desc:
            colnames.append(col[0].lower())
        return remote_data, colnames, literal_controlvalues


class dbconnectorException(Exception):
    def __init__(self, *args):
        self.args = args
        dbtype, connection, error = args
        self.value = 'Connection failed for %s with %s (%s)' % (
            dbtype, connection, error)
